/***************************************************************************
 *
 * Copyright (C) 2001 International Business Machines
 * All rights reserved.
 *
 * This file is part of the GPFS mmfslinux kernel module.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer. 
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution. 
 *  3. The name of the author may not be used to endorse or promote products 
 *     derived from this software without specific prior written
 *     permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *************************************************************************** */
/*
 * Definitions for system dependent types, Linux version
 *
 * Contents:
 *   system dependent types
 *
 */

/*
 * $Id: cxiTypes-plat.h,v 1.82 2001/10/03 14:46:18 dcraft Exp $
 *
 * $Log: cxiTypes-plat.h,v $
 * Revision 1.82  2001/10/03 14:46:18  dcraft
 * First attempt to bring us up to 2.4.9 and 2.4.10
 *
 * Revision 1.81  2001/09/22 20:10:38  dcraft
 * Remove kiobufs from cxiKernelIODescriptor_t.  Use temporary
 * kiobufs for map/unmap.   Remove dead code and dead comments
 * in portability layer and update readmes and license.
 * Fix traceback to appear in mmfs.log file.
 *
 * Revision 1.80  2001/09/20 17:53:53  gjertsen
 * Set CXINODE_SIZE appropriately for IA64. Clean up ifdefs.
 *
 * Revision 1.79  2001/09/20 05:58:52  wsawdon
 * Renamed va_xperm to va_xinfo and definef bit flags to allow var to be
 * shared between extended permissions and snapLatest entries.
 *
 * Revision 1.78  2001/08/30 23:22:37  yuri
 * On AIX 5.1, ino_t is int32 in 32b mode, and int64 in 64b.  In cxiVattr_t,
 * va_serialno field has to be of the same type as system ino_t since it's
 * typecast to vattr_t on AIX.  GPFS internal inode type must be int32.
 * Create a new type cxiInoSys_t, that corresponds to ino_t on AIX and stays
 * Int32 on Linux, declare va_serialno as cxiInoSys_t.
 *
 * Revision 1.77  2001/08/18 02:14:39  tee
 * Improve matching between AIX and Linux mmap code to prepare for source file
 * merge.
 *
 * Revision 1.76  2001/08/10 00:41:44  wyllie
 * Use lock indices instead of lock numbers when calling cxiBlockingMutexInit,
 * to simplify statistics gathering.  Add call to reset or read statistics
 * for cxiBlockingButex_ts from the kernel.
 *
 * Revision 1.75  2001/08/06 23:39:59  wyllie
 * Count number of acquires and contention for cxiBlockingMutex_ts under
 * ifdef INSTRUMENT_LOCKS.
 *
 * Revision 1.74  2001/07/31 20:52:07  tee
 * Defect 344618 - Mapping file in soft-mounted directory can cause kernel
 * assertion failure.  Mmap cannot assume that first vnode pointer in gnode is
 * the one used for mapping.  Change page fault code not to use vnode at all.
 * Instead. save VFS data pointer in gpfsNode_t at mapping time.  Also, since
 * paging code can no longer put a hold on vnode, fix synchronization between
 * pager and mmap termination to ensure that termination won't finish until
 * all outstanding paging requests are complete.
 *
 * Revision 1.73  2001/07/19 23:25:33  dcraft
 * Modified linux trace to allow non blocking trace record
 * writes (format is TRACE?N).  New gpfs swapd process created
 * which is responsible for reclaiming inodes (5 percent every
 * time it runs).  Marked all our inodes so that they would be
 * ignored by linux kswapd.  Added "unused" inode to inode
 * cache that could be used as a signal that linux kswapd is
 * running and kick off gpfs swapd.  Added means to ignore attempts
 * to kill mmfsd by kswapd if the system gets low on memory.
 * All done in an attempt to avoid kswapd premature wakeup on simple
 * locks and mutexes.
 *
 * Revision 1.72  2001/06/25 21:23:17  tee
 * Make direct I/O work on Linux.
 *
 * Revision 1.71  2001/06/11 20:01:50  eshel
 * Make sure '<<' is done before '-' by adding parentheses around '<<' operation.
 *
 * Revision 1.70  2001/06/08 16:48:57  wyllie
 * Define MNT_NFS3 so fcntl lock code compiles
 *
 * Revision 1.69  2001/05/30 20:42:30  wyllie
 * Purge Linux inode when give up token for a file in the stat cache.
 *
 * Revision 1.68  2001/05/25 14:45:33  gjertsen
 * Tweak things for picky compiler.
 *
 * Revision 1.67  2001/05/12 18:38:34  schmuck
 * Fix problem with NFS returning spurious errors under load.
 *
 * Revision 1.66  2001/05/11 20:13:10  eshel
 * Cleanup some GPFS_LINUX ifdefs.
 *
 * Revision 1.65  2001/05/09 00:43:48  wsawdon
 * Fixed bugs in linux readdir returning internal inode number
 * instead of external snapshot/inode numbers.
 *
 * Revision 1.64  2001/05/02 20:04:48  wyllie
 * Put definition and all uses of the extInodeNum field of the cxiNode_t
 * under ifdef SNAPSHOTS.
 *
 * Revision 1.63  2001/05/01 20:06:56  wsawdon
 * Fix for Raleigh defect 1947.
 * Linux kernel code must locate gnodes in the same
 * way that it locates its inode structures.
 *
 * Revision 1.62  2001/04/23 18:14:46  eshel
 * Call DaemonToDie to get internal dump.
 *
 * Revision 1.61  2001/04/22 16:36:51  dcraft
 * Reimplement wait queue structure to have a chain of linux
 * wait queue heads and only one waiter per head.  This allows
 * us to control exactly which task will wake up.  Previously
 * the OS was free to select any task on the wait queue head.
 * This gave us incorrect semantics for "wakeup with result"
 * and resulted in crashes stating unexpected EINTR from wait.
 *
 * Revision 1.60  2001/04/20 22:52:41  eshel
 * Add createThreadId to keep track of thread that did the create.
 *
 * Revision 1.59  2001/04/03 15:31:20  dcraft
 * Update sizes to largest possible (including debug sizes)
 *
 * Revision 1.58  2001/03/28 19:21:06  dcraft
 * Updates to allow building on 2.4.2-0.1.28 kernel (Redhat 7.1)
 * Define LINUX_KERNEL_VERSION 2049902 and don't define KIOBUF_STUFF.
 *
 * Revision 1.57  2001/03/08 01:24:21  dcraft
 * Syntactic sugar for more strict g++ compiler.  No functional change.
 *
 * Revision 1.56  2001/03/07 19:08:40  wyllie
 * Make assigment operators return *this
 *
 * Revision 1.55  2001/03/05 23:28:26  dcraft
 * Modify inode and gpfsNode reference management.  Inode is now acquired
 * during gpfsNode creation and must be released via cxiPutOSNode().
 * (documented in gpfs database).  Add "mmfsadm dump vnodes" for producing
 * trace info on all held inodes.
 *
 * Revision 1.54  2001/03/01 20:47:04  radhak
 * Added mmapFlush to  cxiNode_t
 *
 * Revision 1.53  2001/02/28 17:26:58  wyllie
 * Declare _asm_swap64 using EXTERNC so this file can be included from C
 * as well as C++ files.
 *
 * Revision 1.52  2001/02/28 17:05:16  wyllie
 * Implement _asm_swap64 as an external assembler module, since gcc generates
 * incorrect code for the inline assembler version in some cases.
 *
 * Revision 1.51  2001/02/19 23:08:00  eshel
 * Add code for restore start/end.
 *
 * Revision 1.50  2001/02/17 02:37:19  wyllie
 * C++ classes that store data internally in big-endian format.  Operators
 * in these classes automatically do the necessary byte swapping without
 * (too many) changes to the source code.
 *
 * Revision 1.49  2001/02/15 17:18:21  eshel
 * Use adjusted inode numbers for snapshot inodes.
 *
 * Revision 1.48  2001/02/08 18:04:29  schmuck
 * Tweak fast path through gpfs_i_permission for files that don't have
 * extended acls: instead of doing the check in gpfs_i_permission,
 * set inode_operations::permission to NULL so Linux will do the check
 * without invoking gpfs_i_permission.
 * No functional change.
 *
 * Revision 1.47  2001/01/08 20:49:05  gjertsen
 * Minor code cleanup. No functional changes.
 *
 * Revision 1.46  2000/12/28 22:02:55  gjertsen
 * Fix linux build break.
 *
 * Revision 1.45  2000/12/21 23:42:44  eshel
 * Always define GPFS_LINUX_SEM_SIZE
 *
 * Revision 1.44  2000/12/15 13:57:21  gjertsen
 * Clean up documentation.
 *
 * Revision 1.43  2000/12/14 20:55:30  wyllie
 * Make cxiWaitEvent be allocated inside of Simple_lock object, instead of
 * malloc'ed elsewhere.  Record owner of Simple_lock as a stack address instead
 * of a thread id.  Shorten state word of a Simple_lock to 32 bits on ia64.
 *
 * Revision 1.42  2000/12/13 18:35:53  wyllie
 * Move definition of cxiBlockingMutex_t type to cxiTypes-plat.h.
 *
 * Revision 1.41  2000/12/09 22:49:36  schmuck
 * minor tweaks
 *
 * Revision 1.40  2000/12/08 02:43:11  schmuck
 * The d_revalidate operation is supposed to check whether the cached
 * directory entry is still valid, not whether the file it refers to
 * still exists.
 *
 * Revision 1.39  2000/12/07 21:38:28  schmuck
 * Add a call to invalidate stat information cached in the Linux inode when
 * the inode token is relinquished/downgraded, so that gpfs_i_revalidate
 * optimization works correctly with multiple nodes.
 * Add similar optimization for gpfs_i_permission.
 * Remove NB_PERF ifdef from this code.
 *
 * Revision 1.38  2000/11/30 21:02:17  gjertsen
 * Add platform specific headers for kernel extension functions to help avoid
 * further #ifdef proliferation.
 *
 * Revision 1.37  2000/11/22 22:28:04  wyllie
 * Use BEGIN_FAR_CODE/END_FAR_CODE macros to generate code that is rarely
 * executed far away from the common code, to improve Icache behavior on
 * i386.  Modify FAR_TRACE asm code for clarity.
 *
 * Revision 1.36  2000/11/13 16:46:20  wyllie
 * Rename fields of simple_lock to make them easier to find with cscope
 *
 * Revision 1.35  2000/11/10 18:17:09  wyllie
 * Use REGPARMS on routine declaration to cause GCC to pass parameters in
 * registers.
 *
 * Revision 1.34  2000/11/07 22:04:50  schmuck
 * cxiDirent_t no longer needed
 *
 * Revision 1.33  2000/11/07 19:25:14  schmuck
 * Use same Linux-style filldir interface for readdir on Linux and aix.
 *
 * Revision 1.32  2000/11/06  19:56:41  gjertsen
 * Linux code cleanup and put in build safeguards.
 *
 * Revision 1.31  2000/11/02 19:47:00  gjertsen
 * Linux code split.
 *
 * Revision 1.30  2000/10/30  20:49:37  radhak
 * Linux mmap support
 *
 * Revision 1.29  2000/10/25  20:17:50  dcraft
 * Add cxiSigContext_t
 *
 * Revision 1.28  2000/10/19  19:09:29  dixonbp
 * Add NFSData* to cxiNode_t for linux.
 *
 * Revision 1.27  2000/09/25  16:54:07  schmuck
 * Experimental code for improving NetBench performance.
 * Compile with -DNB_PERF to activate.
 *
 * Revision 1.26  2000/09/20  17:55:49  wyllie
 * Rename VREG to cxiVREG, etc. to avoid conflicts with some Linus patches.
 *
 * Revision 1.25  2000/09/13  20:38:10  dcraft
 * Translate linux open flags to correct FWRITE/FREAD value.
 * Fix mode complaints during linux compilation.
 *
 * Revision 1.24  2000/08/29  18:33:17  dcraft
 * Produce mmfs module.
 *
 * Revision 1.23  2000/08/28  14:15:27  gjertsen
 * Use C wrapper functions for C++ class methods in
 * cxi interface. Convert all cxi functions to C interface.
 *
 * Revision 1.22  2000/08/21  22:16:15  dcraft
 * Create cxiDev_t type that is based on user level dev_t.  Provide
 * mapping functions between kernel, user, and inode device field.
 * Fix NLS yes/no query.
 *
 * Revision 1.21  2000/08/21  15:41:47  gjertsen
 * Add in new atomic op data type (atomic_l).
 *
 * Revision 1.20  2000/08/10  00:02:58  wyllie
 * Buffer management for Linux, phase III: physical disk I/O.  I/O is done
 * synchronously by kxStartIO.  For well-aligned buffers (full blocks starting
 * at offset 0), uses the kiobufs already built.  For other I/O, builds a
 * temporary kiobuf to pass to brw_kiovec.
 *
 * Revision 1.19  2000/08/07  22:33:46  dcraft
 * Use new cxiGetTOD() time of day function for second,
 * nanosecond needs.  Define 32 bit time types and start
 * using in place of current on disk time_t types (necessary
 * for 64 bit).  Use HiResTime where possible.
 *
 * Revision 1.18  2000/08/07  13:42:15  gjertsen
 * Replace use of vmhandle_t with cxiVmhandle_t. These changes are primarily
 * IA64 Linux oriented.
 *
 * Revision 1.17  2000/08/02  14:56:03  dcraft
 * Typedef Ptrdiff correction from int to long for 64bit
 * Correct quota flags for root user override
 *
 * Revision 1.16  2000/07/31  21:12:54  dcraft
 * Define cxiDirent_t and move some defines out of gpfslinux.h
 * to cxiTypes.h
 *
 * Revision 1.15  2000/07/20  15:49:35  dcraft
 * Abstract event words for OS independence
 *
 * Revision 1.14  2000/07/11  19:31:55  dcraft
 * gpfsNode_t::findOrCreate no longer creates an OS specific node on
 * linux.  This caused coding problems since gpfsNode release cannot call
 * iput() itself nor can it decrement the i_count.  The inode is gotten
 * at the vfs specific layer thereby guaranteeing that linux will be
 * responsible for the iput() and i_count decrement.
 *
 * Revision 1.13  2000/07/11  16:36:08  wyllie
 * Use cxiUio_t instead of struct uio.  Use cxiUiomove instead of uiomove.  Use
 * CXI_READ instead of UIO_READ, etc.
 *
 * Revision 1.12  2000/06/30  16:24:02  wyllie
 * Add typedef for struct cxiNode_t.
 *
 * Revision 1.11  2000/06/28  12:47:44  gjertsen
 * Now use eflock_t in platform independant code due to problem with
 * compiling struct eflock when eflock is already typedef'ed to a struct.
 * Now use cxiSetPri and cxiGetPri instead of native AIX calls setpri and getpri.
 *
 * Revision 1.10  2000/06/23  20:31:41  dcraft
 * Remove sysdep.h and ModeBits.h.  Pare down some OS specific
 * header includes.  Use cxiMode_t instead of mode_t due to
 * linux short mode_t in kernel.
 *
 * Revision 1.9  2000/06/20  23:53:52  dcraft
 * Remove OS specifics from mmfsnode.C and mmfsnode.h. Remove sharkvfs.h.
 * Remove unnecessary OS specific ifdefs of header files.
 *
 * Revision 1.8  2000/06/16  20:33:18  dcraft
 * Split MMFSNode into cxiNode_t containing OS dependent items
 * (such as gnode) and gpfsNode_t OS independent items.  Acquire
 * inodes during creation of gpfsNodes.  Remove __FULL_PROTO and
 * system prototypes from all C files.  Turn on __FULL_PROTO in
 * site.mcr.
 *
 * Revision 1.7  2000/06/12  21:42:42  dixonbp
 * Move cxiFlock_t from cxiTypes-plat to cxiTypes.
 *
 * Revision 1.6  2000/06/12  19:32:43  dcraft
 * Rename SharkPrivateVFSData to gpfsVFSData_t and split off into OS independent
 * vfsdata.h.   OS dependent layer relies only on forward declaration gpfsVFSData_t
 * since it uses it only as a pointer (hence the non cxi name).
 *
 * Revision 1.5  2000/06/12  13:20:29  gjertsen
 * Change type names to cxi convention (cxiPid_t and cxiKey_t)
 *
 * Revision 1.4  2000/06/08  13:06:37  gjertsen
 * Introduce cxi types used at user-level and kernel extension
 * interfaces for xmem and uio. Purged out gpfs_flock with cxiFlock_t.
 *
 * Revision 1.3  2000/06/06  20:05:10  dixonbp
 * Add typedef for eflock.
 *
 * Revision 1.2  2000/06/05  19:01:33  dcraft
 * Complete mount callbacks within the OS dependent kernel extension.
 * Record vfs number in gpfsVfsData_t.
 *
 * Revision 1.1  2000/05/30  21:26:05  wyllie
 * Move types from cxiSystemTypes.h to cxiTypes.h.  Use cxi prefix instead of kxi.
 *
 */

#ifndef _h_cxiTypes_plat
#define _h_cxiTypes_plat

/* Use system types.h except in portability kernel code */
#if !defined(GPFS_GPL) || !defined(_KERNEL)
#include <sys/types.h>
#endif


/* Suffix of a routine declaration to tell GCC to pass parameters in
   registers */
#ifdef GPFS_ARCH_I386
#define REGPARMS __attribute__((regparm(3)))
#else
#define REGPARMS
#endif

/* Inline functions to byte-swap integers of various sizes.  These use
   assembler helpers on some architectures.  If the input parameter is
   a constant, the GCC compiler will generate better code by compile-time
   optimization of a (complicated) expression involving shifts and masks
   rather than calling the inline asm code. */
#ifdef GPFS_ARCH_I386
  static inline const UInt16 _asm_swap16(UInt16 val)
  { /* "q" means any of regs AX-DX, which are the only ones that can be
       addressed as bytes */
    __asm__("xchgb %b0,%h0  # _asm_swap16" \
            : "=q" (val)                   \
            :  "0" (val));
    return val;
  }
  static inline UInt16 ByteSwap16(UInt16 val)
  {
    return __builtin_constant_p(val)
      ? _ByteSwap16(val)
      : _asm_swap16(val);
  }
#else
  static inline UInt16 ByteSwap16(UInt16 val)
  {
    return _ByteSwap16(val);
  }
#endif


#ifdef GPFS_ARCH_I386
  static inline const UInt32 _asm_swap32(UInt32 val)
  {
    __asm__("bswap %0  # _asm_swap32"   \
            : "=r" (val)                \
            : "0" (val));
    return val;
  }
  static inline UInt32 ByteSwap32(UInt32 val)
  {
    return __builtin_constant_p(val)
      ? _ByteSwap32(val)
      : _asm_swap32(val);
  }
#else
  static inline UInt32 ByteSwap32(UInt32 val)
  {
    return _ByteSwap32(val);
  }
#endif


#ifdef GPFS_ARCH_I386
  /* Use an external assembler routine to do the swap, since inline
     assembler either generates poor code or incorrect code in some cases.
     The inline asembler is preserved in tasking/i386/swap.S. */
  EXTERNC const UInt64 _asm_swap64(UInt64 val) REGPARMS;
  static inline UInt64 ByteSwap64(UInt64 val)
  {
    return __builtin_constant_p(val)
      ? _ByteSwap64(val)
      : _asm_swap64(val);
  }
#else  /* not GPFS_ARCH_I386 */
  static inline UInt64 ByteSwap64(UInt64 val)
  {
    return  _ByteSwap64(val);
  }
#endif  /* GPFS_ARCH_I386 */


/* On little-endian machines, declare classes for big-endian integers
   of various sizes.  Each class provides operator= and integer cast
   operators, so the compiler will insert code to do conversions
   automatically when evaluating expressions involving big-endian
   integers.  When passing a BigEndIntxx as a value parameter to a
   function without a prototype for its arguments, be sure to explicitly
   cast to the corresponding native arithmetic type.  Otherwise, passing
   the class instance by value will result in the wrong byte order in
   the called routine.  This most often occurs with printf and related
   functions.  Notice that the code generated by TRACEn macros already
   contains such an explicit cast, so no case is needed in the source
   code. */
#ifdef __cplusplus
# if defined(BIG_END_ON_DISK) && defined(GPFS_LITTLE_ENDIAN)

    class BigEndInt16
    {
    private:
      Int16 val16;

    public:
      operator Int16() const
        { return ByteSwap16(val16); }
      BigEndInt16 operator=(Int16 newVal16)
        { val16 = ByteSwap16(newVal16); return *this; }
      int operator==(BigEndInt16 rhs) const
        { return val16 == rhs.val16; }
      int operator!=(BigEndInt16 rhs) const
        { return val16 != rhs.val16; }
      BigEndInt16 operator+=(Int16 inc)
        { val16 = ByteSwap16(ByteSwap16(val16)+inc); return *this; }
      BigEndInt16 operator-=(Int16 inc)
        { val16 = ByteSwap16(ByteSwap16(val16)-inc); return *this; }
      BigEndInt16 operator&=(Int16 mask)
        { val16 &= ByteSwap16(mask); return *this; }
      BigEndInt16 operator|=(Int16 mask)
        { val16 |= ByteSwap16(mask); return *this; }
    };

    class BigEndUInt16
    {
    private:
      UInt16 uval16;

    public:
      operator UInt16() const
        { return ByteSwap16(uval16); }
      BigEndUInt16 operator=(UInt16 newVal16)
        { uval16 = ByteSwap16(newVal16); return *this; }
      int operator==(BigEndUInt16 rhs) const
        { return uval16 == rhs.uval16; }
      int operator!=(BigEndUInt16 rhs) const
        { return uval16 != rhs.uval16; }
      BigEndUInt16 operator+=(UInt16 inc)
        { uval16 = ByteSwap16(ByteSwap16(uval16)+inc); return *this; }
      BigEndUInt16 operator-=(UInt16 inc)
        { uval16 = ByteSwap16(ByteSwap16(uval16)-inc); return *this; }
      BigEndUInt16 operator&=(UInt16 mask)
        { uval16 &= ByteSwap16(mask); return *this; }
      BigEndUInt16 operator|=(UInt16 mask)
        { uval16 |= ByteSwap16(mask); return *this; }
    };

    class BigEndInt32
    {
    private:
      Int32 val32;

    public:
      operator Int32() const
        { return ByteSwap32(val32); }
      BigEndInt32 operator=(Int32 newVal32)
        { val32 = ByteSwap32(newVal32); return *this; }
      int operator==(BigEndInt32 rhs) const
        { return val32 == rhs.val32; }
      int operator!=(BigEndInt32 rhs) const
        { return val32 != rhs.val32; }
      BigEndInt32 operator+=(Int32 inc)
        { val32 = ByteSwap32(ByteSwap32(val32)+inc); return *this; }
      BigEndInt32 operator-=(Int32 inc)
      { val32 = ByteSwap32(ByteSwap32(val32)-inc); return *this; }
    };

    class BigEndUInt32
    {
    private:
      UInt32 uval32;

    public:
      operator UInt32() const
        { return ByteSwap32(uval32); }
      BigEndUInt32 operator=(UInt32 newVal32)
        { uval32 = ByteSwap32(newVal32); return *this; }
      int operator==(BigEndUInt32 rhs) const
        { return uval32 == rhs.uval32; }
      int operator!=(BigEndUInt32 rhs) const
        { return uval32 != rhs.uval32; }
      BigEndUInt32 operator+=(UInt32 inc)
        { uval32 = ByteSwap32(ByteSwap32(uval32)+inc); return *this; }
      BigEndUInt32 operator-=(UInt32 inc)
        { uval32 = ByteSwap32(ByteSwap32(uval32)-inc); return *this; }
      BigEndUInt32 operator&=(UInt32 mask)
        { uval32 &= ByteSwap32(mask); return *this; }
      BigEndUInt32 operator|=(UInt32 mask)
        { uval32 |= ByteSwap32(mask); return *this; }
      BigEndUInt32 operator^=(UInt32 mask)
        { uval32 ^= ByteSwap32(mask); return *this; }
    };

    class BigEndInt64
    {
    private:
      Int64 val64;

    public:
      operator Int64() const
        { return ByteSwap64(val64); }
      BigEndInt64 operator=(Int64 newVal64)
        { val64 = ByteSwap64(newVal64); return *this; }
      int operator==(BigEndInt64 rhs) const
        { return val64 == rhs.val64; }
      int operator!=(BigEndInt64 rhs) const
        { return val64 != rhs.val64; }
      BigEndInt64 operator+=(Int64 inc)
        { val64 = ByteSwap64(ByteSwap64(val64)+inc); return *this; }
      BigEndInt64 operator-=(Int64 inc)
        { val64 = ByteSwap64(ByteSwap64(val64)-inc); return *this; }
    };

    class BigEndUInt64
    {
    private:
      UInt64 uval64;

    public:
      operator UInt64() const
        { return ByteSwap64(uval64); }
      BigEndUInt64 operator=(UInt64 newVal64)
        { uval64 = ByteSwap64(newVal64); return *this; }
      int operator==(BigEndUInt64 rhs) const
        { return uval64 == rhs.uval64; }
      int operator!=(BigEndUInt64 rhs) const
        { return uval64 != rhs.uval64; }
      BigEndUInt64 operator+=(UInt64 inc)
        { uval64 = ByteSwap64(ByteSwap64(uval64)+inc); return *this; }
      BigEndUInt64 operator-=(UInt64 inc)
        { uval64 = ByteSwap64(ByteSwap64(uval64)-inc); return *this; }
    };
# else /* BIG_END_ON_DISK and GPFS_LITTLE_ENDIAN */
    typedef Int16  BigEndInt16;
    typedef UInt16 BigEndUInt16;
    typedef Int32  BigEndInt32;
    typedef UInt32 BigEndUInt32;
    typedef Int64  BigEndInt64;
    typedef UInt64 BigEndUInt64;
# endif /* BIG_END_ON_DISK */
# else /* not __cplusplus */
    typedef Int16  BigEndInt16;
    typedef UInt16 BigEndUInt16;
    typedef Int32  BigEndInt32;
    typedef UInt32 BigEndUInt32;
    typedef Int64  BigEndInt64;
    typedef UInt64 BigEndUInt64;
#endif /* __cplusplus */

/* Standard variable argument type omitted with g++ */
#ifdef __cplusplus
typedef void           *va_list;
#endif /* __cplusplus */

/* Generic function that returns an integer */
#ifdef GPFS_ARCH_IA64
/* we use a macro here to get around picky IA64 compiler */
#define GenIntFunc int (*)(...)
#else
typedef int (*GenIntFunc)();
#endif

/* GPFS stores special file device numbers as 32 bit values
 * (the original AIX dev_t size).  The cxiDev_t type is 64 bits
 * since this is the size of the user space dev_t and the prevalent
 * type that we're always presented with from the stat structure.
 * The cxiKernelDev_t is what we're provided with from the linux
 * VFS and hence we map this to a cxiDev_t.
 */
typedef UInt64  cxiDev_t;        /* user space dev_t definition */
#ifdef GPFS_ARCH_IA64
typedef UInt32  cxiKernelDev_t;  /* Kernel space dev_t definition */
#else
typedef UInt16  cxiKernelDev_t;  /* Kernel space dev_t definition */
#endif

typedef void *  cxiVmid_t;  // XXX proper definition waiting on mmap support
typedef UIntPtr cxiVmhandle_t;  // try this for now

/* Time types equivalent to linux definitions of time_t 
 * and suseconds_t.
 */
typedef long int cxiNSec_t;
typedef long int cxiTime_t;

/* Type used as parameter of ATOMIC_ADD, etc.  */
typedef int    *atomic_p;
/* Type used as parameter of ATOMIC_ADDLP, etc.  */
typedef long   *atomic_l;

/* File offset */
typedef long long offset_t;

/* Complex system types */
struct xmem
{
  int x;
};

/* Cross-memory descriptor */
typedef struct
{
  /* Pointer to buffer descriptor object in kernel address space */
  struct cxiKernelIOBufferDesc_t *kibdP;
} cxiXmem_t;

#ifdef _KERNEL
#define CXIXMEM_TO_XMEM(_XMEM, _CXIXMEM) \
{ \
  memcpy(_XMEM, _CXIXMEM, sizeof(cxiXmem_t)); \
};
#endif

/* Structure representing a buffer in user space that needs to be accessed
   from kernel code. */
struct cxiUio_t
{
  struct cxiIovec_t *uio_iov;    /* ptr to array of iovec structs */
  unsigned long      uio_iovcnt; /* #iovec elements remaining to be processed */
  unsigned long      uio_iovdcnt;/* #iovec elements already processed */
  UInt64             uio_offset; /* byte offset in file/dev to read/write */
  unsigned long      uio_resid;  /* #bytes left in data area */
  short              uio_segflg; /* description of which address space in use */
  long               uio_fmode;  /* copy of file modes from open file struct */
  cxiXmem_t         *uio_xmem;   /* dummy field for AIX compatibility */
};

#define UIO_XMEM 0


/* First parameter on return call to cxiFillDir().  This structure
 * describes the operating systems' filldir() routine and any uninterpreted
 * arguments that need to be passed back to it.   The cxiFillDir() routine
 * has a static set of parameters that are passed to it, whereas the 
 * Linux filldir() routine has changed signatures for different kernel
 * levels.  We thus use cxiFillDir() in the portability layer to map
 * the arguments correctly to the Linux filldir().
 */
typedef struct cxiFillDirArg_t 
{
  void *fnP;
  void *argP;
} cxiFillDirArg_t;

/* Callback function to return directory entries.
 * Called by readddir for each directory entry to be returned to the user.
 * Should return zero if the entry still fits in the user's buffer;
 * otherwise a negative value should be returned.  The 'offset' parameter
 * is the logical offset of the current entry, i.e., a seekdir to that
 * offset followed by a readdir will return the same directory entry again. 
 */
typedef int (*cxiFillDir_t)(void *, const char *, int, offset_t, ino_t);
#define CALL_FILLDIR(fnP, argP, nameP, namelen, offset, ino) \
  (*fnP)(argP, nameP, namelen, offset, ino)

#define GPFS_DIR_EOF 0x7FFFFFFF

/* Advisory locking types and defines */
typedef cxiFlock_t eflock_t;
#define MNT_NFS 2
#define MNT_NFS3 18

/* Define analagous kernel types here for now */
/* These should ONLY be used for typecasting kernel service call parms */
typedef cxiPid_t gpfs_Kpid_t;

/* GPFS file types.  These match the AIX vtype definitions from vnode.h */
enum cxiVtype { cxiVNON, cxiVREG, cxiVDIR, cxiVBLK, cxiVCHR, cxiVLNK,
                cxiVSOCK, cxiVBAD, cxiVFIFO, cxiVMPC };
#define cxiVUNDEF cxiVNON   /* undefined is same as nothing */
typedef enum cxiVtype cxiVtype_t;

/* Base VFS node class portion that is OS specific.  
 * Consult gpfsNode_t for full definition 
 */
typedef struct cxiNode_t
{
  void *osNodeP;        /* pointer to operating system specific inode */
  enum cxiVtype nType;  /* directory, link, file, dev */
  cxiVmid_t mapSeg;     /* ??? TBD for linux */
  int channel;          /* MPX channel (AIX) */
  int mapReadCount;     /* count of map for read  */
  int mapWriteCount;    /* count of map for write */
  int readCount;        /* total opens for read   */
  int writeCount;       /* total opens for write  */
  int execCount;        /* total opens for execute */
  int icValid;          /* CXI_IC_XXX flags (see definitions below) */
  int xinfo;            /* Extended info bits (see va_xinfo definition) */
  void *nfsP;           /* nfs information */
  UInt32 mmapFlush         : 1; /* True if nopage/mmFlush in progress */
  UInt32 destroyIfDelInode : 1; /* True if file should be destroyed in 
                                   gpfs_s_delete_inode */
  UInt32 invalidateCalled  : 1; /* True if OSNode on hash chain has been 
                                   invalidated on this pass. */
  UInt32 cxiNode_t_unused  : 29;
  cxiThreadId createRaceLoserThreadId; /* Set if file exist */
} cxiNode_t;

/* Size of a full gpfsNode_t */
#ifdef GPFS_ARCH_I386
#define CXINODE_SIZE 188
#endif /* GPFS_ARCH_I386 */
#ifdef GPFS_ARCH_IA64
#define CXINODE_SIZE 264
#endif /* GPFS_ARCH_IA64 */

/* Access OS dependent portion of cxiNode */
#define GNP_IS_DIR(GNP)       ((GNP)->nType == cxiVDIR)
#define GNP_IS_FILE(GNP)      ((GNP)->nType == cxiVREG)
#define GNP_IS_LINK(GNP)      ((GNP)->nType == cxiVLNK)

#define GNP_IS_STEALABLE(GNP)  ((GNP)->osNodeP == NULL)
#define GNP_RDCOUNT(GNP)       ((GNP)->readCount)
#define GNP_WRCOUNT(GNP)       ((GNP)->writeCount)
#define GNP_EXCOUNT(GNP)       ((GNP)->execCount)
#define GNP_MAP_RDCOUNT(GNP)   ((GNP)->mapReadCount)
#define GNP_MAP_WRCOUNT(GNP)   ((GNP)->mapWriteCount)
#define GNP_MAP_SEG(GNP)       ((GNP)->mapSeg)          /* ??? */
#define GNP_CHANNEL(GNP)       ((GNP)->channel)         /* ??? */
#define GNP_MAP_SEG_INIT(GNP) ((GNP)->mapSeg = NULL)
#define GP_TO_GNP(x) ((gpfsNode_t *)((x)))


#ifdef SNAPSHOTS
/* Generate an external "inode" number to uniquely identify 
   a file stored within a snapshot. */
#define GET_EXTERNAL_INODENUM(iNum,extSnapId) ((extSnapId << 27) | iNum)
#define GET_INUM_FROM_EXTERNAL_INUM(extINum)   ((extINum) & ((1<<27) - 1))
#define GET_SNAPID_FROM_EXTERNAL_INUM(extINum) ((extINum) >> 27)
#endif

/* Maximum size defined within Linux kernel for spinlock_t
 * We abstract this class to just a hunk of storage and let the OS
 * specific piece handle the implementation.
 */
#ifdef GPFS_ARCH_I386
#define SPINLOCK_T_SIZE 4
#endif /* GPFS_ARCH_I386 */
#ifdef GPFS_ARCH_IA64
#define SPINLOCK_T_SIZE 4
#endif /* GPFS_ARCH_IA64 */

typedef struct cxiWaitList_t
{
  struct cxiWaitList_t *nextP;
  struct cxiWaitList_t *prevP; 
} cxiWaitList_t;

typedef struct cxiWaitEvent_t 
{
  /* List of wait elements. Wakeup is FIFO ordered with entries
   * entering the list at prevP and the first to be awaken at 
   * nextP.
   */
  cxiWaitList_t waitList;

  char lword[SPINLOCK_T_SIZE];

} cxiWaitEvent_t;

#define LOCK_ALLOC_PIN   0 /* not used */
#define LOCK_ALLOC_PAGED 1 /* not used */

/* Needs to be changed to cxi name XXX */
typedef struct Simple_lock 
{
  int slCount;	
  unsigned int slState;
  /* Pointer into stack of owning thread.  This is sufficient to compute
     which thread owns the mutex. */
  char* slOwnerP;
  cxiWaitEvent_t slEvent;
} Simple_lock;


/* Kernel-only mutexes.  Largest possible size (including debug) */
#ifdef GPFS_ARCH_I386
#define GPFS_LINUX_SEM_SIZE 36
#endif /* GPFS_ARCH_I386 */
#ifdef GPFS_ARCH_IA64
#define GPFS_LINUX_SEM_SIZE 56
#endif /* GPFS_ARCH_IA64 */
typedef struct cxiBlockingMutex_t
{
  /* Space to store a Linux struct semaphore.  If a semaphore does not fit
     in the space provided, the implementation should use this space as a
     pointer to a dynamically-allocated semaphore. */
  char bmSem[GPFS_LINUX_SEM_SIZE];

  /* Pointer into stack of owning thread.  This is sufficient to compute
     which thread owns the mutex. */
  char* bmOwnerP;

  /* Index of the name of this mutex */
  int lockNameIndex;
} cxiBlockingMutex_t;


/* Structure encapsulating information about a particular physical disk */
struct cxiDiskInfo_t
{
  Int64 totalSectors;      /* number of sectors on disk */
  int sectorSize;          /* size of each physical sector in bytes */
};

typedef struct ucontext cxiSigContext_t;
typedef struct siginfo cxiSigInfo_t;

/* Type that should always correspond to the system ino_t */
typedef Int32 cxiInoSys_t;


/* Define macros to cause code to be placed far away from the spot
   where it normally would be generated.  This is typically used on
   whichever branch of an if is expected not to be executed in the common
   case.  Moving rarely executed code away from the mainline code can help
   reduce I-cache misses, and therefore improve performance.  Every
   BEGIN_FAR_CODE directive must be matched by an END_FAR_CODE directive,
   and they may not be nested.  If FAR_CODE is not #defined, these macros
   are NOOPs.  Use of the FAR_CODE option is not compatible with -g, and
   requires compiler options -fno-exceptions and -fno-defer-pop to insure
   that correct code is generated. */
#ifdef FAR_CODE
# ifdef GPFS_ARCH_I386
#   ifndef FAR_TRACE
#     error "-DFAR_CODE requires -DFAR_TRACE"
#   endif
#   define BEGIN_FAR_CODE __asm__(                                \
                                  " jmp 9f\n"                     \
                                  ".section .text.far,\"ax\"\n"   \
                                  "9:\n"                          \
                                 );
#   define END_FAR_CODE  __asm__(                                 \
                                 " jmp 8f\n"                      \
                                  ".section .text,\"ax\"\n"   \
                                  "8:\n"                          \
                                );
# else
#   define BEGIN_FAR_CODE
#   define END_FAR_CODE
# endif  /* GPFS_ARCH_I386 */
#else
# define BEGIN_FAR_CODE
# define END_FAR_CODE
#endif  /* FAR_TRACE */

#endif /* _h_cxiTypes_plat */
